/**
 * AdHub Legacy SDK - Compatibility Layer for 4wnetwork tags
 * @version 1.0.0
 * @date 2025-11-17
 * @author AdHubMedia
 * @license PROPRIETARY
 *
 * Replaces: https://static.4wnetwork.com/js/sdk.min.js
 * New URL: https://lib.adhubmedia.com/sdk/sdk.min.js
 */


// ============================================================================
// src/config/defaults.js
// ============================================================================

/**
 * AdHub Legacy SDK - Default Configuration
 *
 * Configurazioni di default per il sistema
 *
 * @version 1.0.0
 */

(function(window) {
  'use strict';

  /**
   * Configurazione globale di default
   */
  const DEFAULT_CONFIG = {
    // Debug mode
    debug: false,

    // URL base per bundle AdHub Legacy (dedicato ai vecchi tag 4wnetwork)
    adhubBundleUrl: 'https://lib.adhubmedia.com/dist/legacy_adhubmedia/adhub-legacy_adhubmedia-no-cmp.min.js',

    // Versione bundle per cache busting (formato: DDMMYYYYHHMM o timestamp)
    // Se null, non aggiunge parametro version
    bundleVersion: null,  // Esempio: '131120251444' o Date.now()

    // Site code di default (può essere sovrascritto)
    defaultSiteCode: 'legacy_adhubmedia',

    // Auto-load bundle AdHub
    autoLoadBundle: true,

    // Aggiungi stile di base ai placeholder
    addBaseStyles: true,

    // Stile base per placeholder
    placeholderBaseStyle: 'margin: 10px auto; text-align: center; min-height: 50px;',

    // Prefix per ID placeholder generati
    placeholderPrefix: 'adhub-legacy-',

    // Key-value personalizzati aggiunti a tutti gli ad units
    globalKeyValues: {
      'tag_old_platform': 'true',
      'sdk_version': 'legacy-1.0.0'
    },

    // Mapping codici IC (index → key)
    icMapping: {
      0: 'legacy_publisher',
      1: 'legacy_zone',
      2: 'legacy_placement',
      3: 'legacy_param'
    },

    // Timeout per caricamento bundle (ms)
    bundleLoadTimeout: 10000,

    // Ritenta caricamento bundle in caso di errore
    retryBundleLoad: true,

    // Numero massimo di tentativi
    maxRetries: 3,

    // Delay tra tentativi (ms)
    retryDelay: 2000,

    // Inserisci placeholder inline (nella posizione dello script) o append al body
    inlineInsertion: true,

    // Se true, preserva lo script originale nel DOM (per debug)
    preserveOriginalScript: false,

    // Tracking eventi
    tracking: {
      enabled: true,
      endpoint: null, // Se null, usa console.log
      events: {
        tagParsed: true,
        placeholderCreated: true,
        bundleLoaded: true,
        adRendered: false,
        errors: true
      }
    },

    // Gestione errori
    errorHandling: {
      // Mostra errori in console
      logErrors: true,
      // Mostra messaggi di errore nei placeholder (solo in debug mode)
      showErrorsInPlaceholder: false,
      // Fallback: se parsing fallisce, prova comunque a creare placeholder generico
      createFallbackPlaceholder: true
    },

    // Responsive behavior
    responsive: {
      // Nascondi automaticamente placeholder non adatti al device
      autoHide: true,
      // Breakpoints
      breakpoints: {
        mobile: 768,
        tablet: 1024,
        desktop: 1280
      }
    },

    // Lazy loading
    lazyLoad: {
      enabled: false, // Disabilitato di default, gestito da AdHub
      threshold: 200  // px prima di entrare nel viewport
    }
  };

  /**
   * Configurazione specifica per formato
   */
  const FORMAT_CONFIGS = {
    display: {
      defaultStyle: 'display: block; margin: 10px auto;',
      minHeight: '50px',
      autoResize: false
    },
    native: {
      defaultStyle: 'display: block; width: 100%;',
      minHeight: '100px',
      autoResize: true
    },
    video: {
      defaultStyle: 'display: block; margin: 10px auto; background: #000;',
      minHeight: '200px',
      autoResize: false,
      // Attributi specifici per video
      videoAttributes: {
        'data-adhub-video': 'in-article',
        'data-video-autoplay': 'false',
        'data-video-muted': 'true'
      }
    }
  };

  /**
   * Messaggi e labels
   */
  const MESSAGES = {
    loading: 'Caricamento annuncio...',
    error: 'Errore caricamento annuncio',
    noAds: 'Nessun annuncio disponibile',
    blocked: 'Annuncio bloccato (AdBlocker rilevato)'
  };

  /**
   * API pubblica
   */
  window.AdHubLegacyDefaults = {
    /**
     * Ottieni config completa
     */
    getConfig: function() {
      return DEFAULT_CONFIG;
    },

    /**
     * Ottieni config per formato specifico
     */
    getFormatConfig: function(format) {
      return FORMAT_CONFIGS[format] || FORMAT_CONFIGS.display;
    },

    /**
     * Merge config personalizzata con default
     */
    mergeConfig: function(customConfig) {
      const merged = {};

      // Copia config di default
      Object.keys(DEFAULT_CONFIG).forEach(function(key) {
        if (typeof DEFAULT_CONFIG[key] === 'object' && !Array.isArray(DEFAULT_CONFIG[key])) {
          merged[key] = Object.assign({}, DEFAULT_CONFIG[key]);
        } else {
          merged[key] = DEFAULT_CONFIG[key];
        }
      });

      // Merge config custom
      if (customConfig) {
        Object.keys(customConfig).forEach(function(key) {
          if (typeof customConfig[key] === 'object' && !Array.isArray(customConfig[key])) {
            merged[key] = Object.assign(merged[key] || {}, customConfig[key]);
          } else {
            merged[key] = customConfig[key];
          }
        });
      }

      return merged;
    },

    /**
     * Ottieni messaggio
     */
    getMessage: function(key) {
      return MESSAGES[key] || '';
    },

    /**
     * Ottieni tutti i messaggi
     */
    getMessages: function() {
      return MESSAGES;
    },

    /**
     * Imposta config globale da window object
     */
    loadFromWindow: function() {
      if (window.AdHubLegacyConfig) {
        return this.mergeConfig(window.AdHubLegacyConfig);
      }
      return DEFAULT_CONFIG;
    },

    /**
     * Debug: mostra configurazione
     */
    debug: function() {
      console.group('⚙️  AdHub Legacy SDK - Configuration');
      console.log('Default Config:', DEFAULT_CONFIG);
      console.log('Format Configs:', FORMAT_CONFIGS);
      console.log('Messages:', MESSAGES);
      console.groupEnd();
    }
  };

})(window);


// ============================================================================
// src/config/dimensions.js
// ============================================================================

/**
 * AdHub Legacy SDK - Dimensions Mapping Configuration
 *
 * Mappa le dimensioni dei vecchi tag 4wnetwork ai nuovi formati AdHub
 *
 * @version 1.0.0
 */

(function(window) {
  'use strict';

  /**
   * Mapping dimensioni → formato
   * Ogni dimensione può avere:
   * - type: display, native, video
   * - category: banner, skyscraper, rectangle, etc.
   * - placeholder: template ID per il placeholder AdHub
   * - multiple: se true, supporta istanze multiple (aggiunge counter)
   */
  const DIMENSION_MAP = {
    // ========================================================================
    // DISPLAY ADS - DESKTOP
    // ========================================================================
    '728x90': {
      type: 'display',
      category: 'leaderboard',
      placeholder: 'adhub-728x90-banner',
      multiple: true,
      description: 'Leaderboard (Desktop)'
    },
    '970x90': {
      type: 'display',
      category: 'leaderboard',
      placeholder: 'adhub-970x90-banner',
      multiple: true,
      description: 'Super Leaderboard (Desktop)'
    },
    '970x250': {
      type: 'display',
      category: 'billboard',
      placeholder: 'adhub-billboard',
      multiple: true,
      description: 'Billboard (Desktop)'
    },
    '300x250': {
      type: 'display',
      category: 'rectangle',
      placeholder: 'adhub-300x250',
      multiple: true,
      description: 'Medium Rectangle'
    },
    '336x280': {
      type: 'display',
      category: 'rectangle',
      placeholder: 'adhub-336x280',
      multiple: true,
      description: 'Large Rectangle'
    },
    '160x600': {
      type: 'display',
      category: 'skyscraper',
      placeholder: 'adhub-160x600-sidebar',
      multiple: true,
      description: 'Wide Skyscraper'
    },
    '120x600': {
      type: 'display',
      category: 'skyscraper',
      placeholder: 'adhub-120x600-sidebar',
      multiple: true,
      description: 'Skyscraper'
    },
    '300x600': {
      type: 'display',
      category: 'halfpage',
      placeholder: 'adhub-300x600-sidebar',
      multiple: true,
      description: 'Half Page'
    },
    '468x60': {
      type: 'display',
      category: 'banner',
      placeholder: 'adhub-468x60-banner',
      multiple: true,
      description: 'Full Banner (Desktop)'
    },
    '234x60': {
      type: 'display',
      category: 'banner',
      placeholder: 'adhub-234x60-banner',
      multiple: true,
      description: 'Half Banner'
    },

    // ========================================================================
    // DISPLAY ADS - MOBILE
    // ========================================================================
    '320x50': {
      type: 'display',
      category: 'mobile-banner',
      placeholder: 'adhub-mobile-banner',
      multiple: true,
      description: 'Mobile Banner'
    },
    '320x100': {
      type: 'display',
      category: 'mobile-banner',
      placeholder: 'adhub-mobile-large',
      multiple: true,
      description: 'Mobile Large Banner'
    },
    '300x50': {
      type: 'display',
      category: 'mobile-banner',
      placeholder: 'adhub-300x50-mobile',
      multiple: true,
      description: 'Mobile Banner Small'
    },

    // ========================================================================
    // NATIVE ADS
    // ========================================================================
    '1x1': {
      type: 'native',
      category: 'native-content',
      placeholder: 'adhub-native',
      multiple: true,
      description: 'Native Ad (Fluid)'
    },
    '0x0': {
      type: 'native',
      category: 'native-content',
      placeholder: 'adhub-native',
      multiple: true,
      description: 'Native Ad (Responsive)'
    },

    // ========================================================================
    // VIDEO ADS
    // ========================================================================
    '640x360': {
      type: 'video',
      category: 'video-player',
      placeholder: 'adhub-video-player',
      multiple: true,
      videoType: 'instream',
      description: 'Video Player 16:9 (Medium)'
    },
    '640x480': {
      type: 'video',
      category: 'video-player',
      placeholder: 'adhub-video-player',
      multiple: true,
      videoType: 'instream',
      description: 'Video Player 4:3 (Medium)'
    },
    '854x480': {
      type: 'video',
      category: 'video-player',
      placeholder: 'adhub-video-player',
      multiple: true,
      videoType: 'instream',
      description: 'Video Player 480p'
    },
    '1280x720': {
      type: 'video',
      category: 'video-player',
      placeholder: 'adhub-video-player',
      multiple: true,
      videoType: 'instream',
      description: 'Video Player 720p HD'
    },
    '1920x1080': {
      type: 'video',
      category: 'video-player',
      placeholder: 'adhub-video-player',
      multiple: true,
      videoType: 'instream',
      description: 'Video Player 1080p Full HD'
    },
    '300x169': {
      type: 'video',
      category: 'video-small',
      placeholder: 'adhub-video-small',
      multiple: true,
      videoType: 'outstream',
      description: 'Video Small (Outstream)'
    },
    '400x300': {
      type: 'video',
      category: 'video-player',
      placeholder: 'adhub-video-player',
      multiple: true,
      videoType: 'outstream',
      description: 'Video Medium (Outstream)'
    }
  };

  /**
   * Fallback per dimensioni non mappate
   */
  const FALLBACK_CONFIG = {
    type: 'display',
    category: 'custom',
    placeholder: 'adhub-custom',
    multiple: true,
    description: 'Custom Size'
  };

  /**
   * Categorie per raggruppamento
   */
  const CATEGORIES = {
    'leaderboard': {
      name: 'Leaderboard',
      devices: ['desktop', 'tablet']
    },
    'billboard': {
      name: 'Billboard',
      devices: ['desktop']
    },
    'rectangle': {
      name: 'Rectangle',
      devices: ['desktop', 'tablet', 'mobile']
    },
    'skyscraper': {
      name: 'Skyscraper',
      devices: ['desktop']
    },
    'halfpage': {
      name: 'Half Page',
      devices: ['desktop']
    },
    'banner': {
      name: 'Banner',
      devices: ['desktop', 'tablet']
    },
    'mobile-banner': {
      name: 'Mobile Banner',
      devices: ['mobile']
    },
    'native-content': {
      name: 'Native Content',
      devices: ['all']
    },
    'video-player': {
      name: 'Video Player',
      devices: ['all']
    },
    'video-small': {
      name: 'Video Small',
      devices: ['all']
    }
  };

  /**
   * API pubblica
   */
  window.AdHubLegacyDimensions = {
    /**
     * Ottieni configurazione per dimensioni specifiche
     * @param {string} dimension - Dimensioni in formato "WIDTHxHEIGHT" (es: "728x90")
     * @returns {Object} Configurazione mappata o fallback
     */
    getConfig: function(dimension) {
      const normalized = this.normalizeDimension(dimension);
      return DIMENSION_MAP[normalized] || FALLBACK_CONFIG;
    },

    /**
     * Normalizza dimensioni da vari formati
     * @param {string|Object} dim - Dimensioni come "728x90" o {width: "728px", height: "90px"}
     * @returns {string} Dimensioni normalizzate "WIDTHxHEIGHT"
     */
    normalizeDimension: function(dim) {
      if (typeof dim === 'string') {
        // Già nel formato corretto
        return dim.toLowerCase().replace(/[^0-9x]/g, '');
      }

      if (typeof dim === 'object') {
        // Formato oggetto {width: "728px", height: "90px"}
        const width = String(dim.width || dim.w || '0').replace(/[^0-9]/g, '');
        const height = String(dim.height || dim.h || '0').replace(/[^0-9]/g, '');
        return width + 'x' + height;
      }

      return '0x0';
    },

    /**
     * Verifica se dimensioni sono supportate
     * @param {string} dimension - Dimensioni da verificare
     * @returns {boolean}
     */
    isSupported: function(dimension) {
      const normalized = this.normalizeDimension(dimension);
      return !!DIMENSION_MAP[normalized];
    },

    /**
     * Ottieni tutte le dimensioni mappate
     * @returns {Object} Mappa completa
     */
    getAll: function() {
      return DIMENSION_MAP;
    },

    /**
     * Ottieni dimensioni per tipo
     * @param {string} type - Tipo: 'display', 'native', 'video'
     * @returns {Array} Lista dimensioni
     */
    getByType: function(type) {
      return Object.keys(DIMENSION_MAP).filter(function(dim) {
        return DIMENSION_MAP[dim].type === type;
      }).map(function(dim) {
        return {
          dimension: dim,
          config: DIMENSION_MAP[dim]
        };
      });
    },

    /**
     * Ottieni dimensioni per categoria
     * @param {string} category - Categoria
     * @returns {Array} Lista dimensioni
     */
    getByCategory: function(category) {
      return Object.keys(DIMENSION_MAP).filter(function(dim) {
        return DIMENSION_MAP[dim].category === category;
      }).map(function(dim) {
        return {
          dimension: dim,
          config: DIMENSION_MAP[dim]
        };
      });
    },

    /**
     * Ottieni info categoria
     * @param {string} category - Nome categoria
     * @returns {Object} Info categoria
     */
    getCategoryInfo: function(category) {
      return CATEGORIES[category] || null;
    },

    /**
     * Debug: mostra tutte le mappature
     */
    debug: function() {
      console.group('📐 AdHub Legacy Dimensions Map');
      console.log('Total mappings:', Object.keys(DIMENSION_MAP).length);

      console.group('Display Ads');
      this.getByType('display').forEach(function(item) {
        console.log(item.dimension, '→', item.config.placeholder, '-', item.config.description);
      });
      console.groupEnd();

      console.group('Native Ads');
      this.getByType('native').forEach(function(item) {
        console.log(item.dimension, '→', item.config.placeholder, '-', item.config.description);
      });
      console.groupEnd();

      console.group('Video Ads');
      this.getByType('video').forEach(function(item) {
        console.log(item.dimension, '→', item.config.placeholder, '-', item.config.description);
      });
      console.groupEnd();

      console.groupEnd();
    }
  };

})(window);


// ============================================================================
// src/core/parser.js
// ============================================================================

/**
 * AdHub Legacy SDK - Tag Parser
 *
 * Parser per vecchi tag 4wnetwork (obj_4w)
 * Estrae e normalizza dati dai tag legacy
 *
 * @version 1.0.0
 */

(function(window) {
  'use strict';

  /**
   * Statistiche parsing
   */
  const stats = {
    total: 0,
    successful: 0,
    failed: 0,
    tags: []
  };

  /**
   * Parser principale
   */
  const AdHubLegacyParser = {
    /**
     * Inizializza parser
     */
    init: function(config) {
      this.config = config || window.AdHubLegacyDefaults.getConfig();
      this.log('🔍 Parser inizializzato');
    },

    /**
     * Processa tutti i tag obj_4w in coda
     * @returns {Array} Array di tag processati
     */
    processAll: function() {
      this.log('📋 Processamento tag obj_4w...');

      // Verifica che obj_4w esista
      if (!window.obj_4w || !Array.isArray(window.obj_4w)) {
        this.log('⚠️  Nessun tag obj_4w trovato');
        return [];
      }

      const tags = [];
      const queue = window.obj_4w.slice(); // Copia array

      this.log('📦 Trovati ' + queue.length + ' tag in coda');

      // Processa ogni tag
      queue.forEach(function(tagData, index) {
        stats.total++;

        try {
          const parsed = this.parseTag(tagData, index);
          if (parsed) {
            tags.push(parsed);
            stats.successful++;
            stats.tags.push(parsed);
          } else {
            stats.failed++;
          }
        } catch (error) {
          this.log('❌ Errore parsing tag #' + index + ':', error);
          stats.failed++;

          // Track error
          if (this.config.tracking.enabled && this.config.tracking.events.errors) {
            this.trackEvent('parsing_error', {
              index: index,
              error: error.message,
              tagData: tagData
            });
          }
        }
      }.bind(this));

      this.log('✅ Parsing completato: ' + stats.successful + ' successi, ' + stats.failed + ' falliti');

      return tags;
    },

    /**
     * Processa singolo tag
     * @param {Object} tagData - Dati del tag da obj_4w
     * @param {number} index - Indice nella coda
     * @returns {Object|null} Tag processato o null se fallito
     */
    parseTag: function(tagData, index) {
      if (!tagData || typeof tagData !== 'object') {
        this.log('⚠️  Tag #' + index + ' non valido (non è un oggetto)');
        return null;
      }

      this.log('🔍 Parsing tag #' + index + ':', tagData);

      // Estrai campi principali
      const cid = tagData.cid || tagData.clientId || null;
      const ic = tagData.ic || tagData.codes || null;
      const dim = tagData.dim || tagData.dimensions || tagData.size || null;

      // Valida campi obbligatori
      if (!cid) {
        this.log('⚠️  Tag #' + index + ' mancante campo "cid"');
        return null;
      }

      if (!dim) {
        this.log('⚠️  Tag #' + index + ' mancante campo "dim"');
        return null;
      }

      // Parse dimensioni
      const dimensions = this.parseDimensions(dim);
      if (!dimensions) {
        this.log('⚠️  Tag #' + index + ' dimensioni non valide:', dim);
        return null;
      }

      // Parse codici IC (targeting)
      const targeting = this.parseIC(ic);

      // Ottieni config dimensioni
      const dimensionConfig = window.AdHubLegacyDimensions.getConfig(dimensions);

      // Cerca script element originale
      const scriptElement = this.findScriptElement(cid);

      // Costruisci oggetto tag processato
      const parsedTag = {
        // Identificatori
        index: index,
        cid: cid,

        // Dimensioni
        dimensions: dimensions,
        dimensionConfig: dimensionConfig,

        // Targeting
        targeting: targeting,

        // Tipo e categoria
        type: dimensionConfig.type,
        category: dimensionConfig.category,

        // Placeholder
        placeholderId: this.generatePlaceholderId(cid, dimensionConfig, index),

        // Script element originale
        scriptElement: scriptElement,

        // Metadata
        timestamp: Date.now(),
        userAgent: navigator.userAgent,

        // Dati raw originali (per debug)
        raw: tagData
      };

      this.log('✅ Tag #' + index + ' processato:', parsedTag);

      // Track evento
      if (this.config.tracking.enabled && this.config.tracking.events.tagParsed) {
        this.trackEvent('tag_parsed', {
          cid: cid,
          type: parsedTag.type,
          dimensions: dimensions
        });
      }

      return parsedTag;
    },

    /**
     * Parse dimensioni da vari formati
     * @param {Object|string} dim - Dimensioni
     * @returns {string|null} Dimensioni normalizzate "WxH" o null
     */
    parseDimensions: function(dim) {
      if (!dim) return null;

      // Se è già una stringa nel formato corretto
      if (typeof dim === 'string') {
        const match = dim.match(/(\d+)\s*x\s*(\d+)/i);
        if (match) {
          return match[1] + 'x' + match[2];
        }
      }

      // Se è un oggetto {width: "728px", height: "90px"}
      if (typeof dim === 'object') {
        let width = dim.width || dim.w;
        let height = dim.height || dim.h;

        if (width && height) {
          // Rimuovi "px" o altri suffissi
          width = String(width).replace(/[^0-9]/g, '');
          height = String(height).replace(/[^0-9]/g, '');

          if (width && height) {
            return width + 'x' + height;
          }
        }
      }

      return null;
    },

    /**
     * Parse codici IC e genera targeting
     * @param {string} ic - Stringa IC (es: "252063;112089;80651;0")
     * @returns {Object} Oggetto con targeting key-values
     */
    parseIC: function(ic) {
      const targeting = Object.assign({}, this.config.globalKeyValues);

      if (!ic) {
        return targeting;
      }

      // Split per separatore (default: ";")
      const codes = String(ic).split(';');

      // Mappa usando icMapping dalla config
      const mapping = this.config.icMapping;

      codes.forEach(function(code, index) {
        const key = mapping[index] || ('legacy_code_' + index);
        const value = code.trim();

        if (value) {
          targeting[key] = value;
        }
      });

      return targeting;
    },

    /**
     * Genera ID placeholder univoco
     * @param {string} cid - Client ID originale
     * @param {Object} dimensionConfig - Config dimensioni
     * @param {number} index - Indice tag
     * @returns {string} ID placeholder
     */
    generatePlaceholderId: function(cid, dimensionConfig, index) {
      // Se dimensione supporta multiple istanze, aggiungi counter
      if (dimensionConfig.multiple) {
        return dimensionConfig.placeholder + '-' + index;
      }

      // Altrimenti usa placeholder base
      return dimensionConfig.placeholder;
    },

    /**
     * Trova script element originale nel DOM
     * @param {string} cid - Client ID
     * @returns {HTMLElement|null} Script element o null
     */
    findScriptElement: function(cid) {
      if (!cid) return null;

      // Cerca per ID
      const byId = document.getElementById(cid);
      if (byId && byId.tagName === 'SCRIPT') {
        return byId;
      }

      // Cerca per attributo id in tutti gli script
      const scripts = document.querySelectorAll('script[id]');
      for (let i = 0; i < scripts.length; i++) {
        if (scripts[i].id === cid) {
          return scripts[i];
        }
      }

      return null;
    },

    /**
     * Ottieni statistiche parsing
     * @returns {Object} Statistiche
     */
    getStats: function() {
      return {
        total: stats.total,
        successful: stats.successful,
        failed: stats.failed,
        successRate: stats.total > 0 ? ((stats.successful / stats.total) * 100).toFixed(2) + '%' : '0%',
        tags: stats.tags
      };
    },

    /**
     * Reset statistiche
     */
    resetStats: function() {
      stats.total = 0;
      stats.successful = 0;
      stats.failed = 0;
      stats.tags = [];
    },

    /**
     * Track evento (se tracking abilitato)
     */
    trackEvent: function(eventName, data) {
      if (!this.config.tracking.enabled) return;

      const event = {
        name: eventName,
        timestamp: Date.now(),
        data: data
      };

      // Se c'è un endpoint, invia
      if (this.config.tracking.endpoint) {
        // TODO: implementare invio a endpoint
        this.log('📊 Track:', event);
      } else {
        // Altrimenti log in console
        this.log('📊 Track:', event);
      }
    },

    /**
     * Helper per logging
     */
    log: function() {
      if (this.config && this.config.debug) {
        const args = Array.prototype.slice.call(arguments);
        console.log.apply(console, ['[AdHub Legacy Parser]'].concat(args));
      }
    },

    /**
     * Debug: mostra info parser
     */
    debug: function() {
      console.group('🔍 AdHub Legacy Parser - Debug');
      console.log('Config:', this.config);
      console.log('Stats:', this.getStats());
      console.log('obj_4w queue:', window.obj_4w);
      console.groupEnd();
    }
  };

  // Esporta globalmente
  window.AdHubLegacyParser = AdHubLegacyParser;

})(window);


// ============================================================================
// src/core/placeholder.js
// ============================================================================

/**
 * AdHub Legacy SDK - Placeholder Generator
 *
 * Genera e inserisce placeholder nel DOM per gli ad units
 * Supporta inserimento inline o append al body
 *
 * @version 1.0.0
 */

(function(window) {
  'use strict';

  /**
   * Statistiche placeholder creati
   */
  const stats = {
    created: 0,
    inserted: 0,
    failed: 0,
    placeholders: []
  };

  /**
   * Generator principale
   */
  const AdHubLegacyPlaceholder = {
    /**
     * Inizializza generator
     */
    init: function(config) {
      this.config = config || window.AdHubLegacyDefaults.getConfig();
      this.log('📦 Placeholder Generator inizializzato');
    },

    /**
     * Crea placeholder per tutti i tag processati
     * @param {Array} tags - Array di tag processati dal parser
     * @returns {Array} Array di placeholder creati
     */
    createAll: function(tags) {
      this.log('🏗️  Creazione placeholder per ' + tags.length + ' tag...');

      const placeholders = [];

      tags.forEach(function(tag) {
        try {
          const placeholder = this.create(tag);
          if (placeholder) {
            placeholders.push(placeholder);
            stats.created++;
            stats.placeholders.push(placeholder);
          } else {
            stats.failed++;
          }
        } catch (error) {
          this.log('❌ Errore creazione placeholder per tag:', tag.cid, error);
          stats.failed++;
        }
      }.bind(this));

      this.log('✅ Creati ' + stats.created + ' placeholder, ' + stats.failed + ' falliti');

      return placeholders;
    },

    /**
     * Crea singolo placeholder
     * @param {Object} tag - Tag processato
     * @returns {Object|null} Placeholder creato o null
     */
    create: function(tag) {
      if (!tag || !tag.placeholderId) {
        this.log('⚠️  Tag non valido per creazione placeholder');
        return null;
      }

      this.log('🏗️  Creazione placeholder:', tag.placeholderId);

      // Verifica se placeholder esiste già
      if (document.getElementById(tag.placeholderId)) {
        this.log('⚠️  Placeholder già esistente:', tag.placeholderId);
        return null;
      }

      // Crea elemento div
      const div = document.createElement('div');
      div.id = tag.placeholderId;

      // Aggiungi attributi data per tracking e debug
      div.setAttribute('data-adhub-legacy', 'true');
      div.setAttribute('data-legacy-cid', tag.cid);
      div.setAttribute('data-legacy-type', tag.type);
      div.setAttribute('data-legacy-dimensions', tag.dimensions);

      // Aggiungi targeting come data attributes
      if (tag.targeting) {
        Object.keys(tag.targeting).forEach(function(key) {
          div.setAttribute('data-targeting-' + key, tag.targeting[key]);
        });
      }

      // Applica stili
      this.applyStyles(div, tag);

      // Aggiungi classe CSS
      div.className = 'adhub-legacy-placeholder adhub-type-' + tag.type;

      // Placeholder object
      const placeholder = {
        id: tag.placeholderId,
        element: div,
        tag: tag,
        inserted: false,
        timestamp: Date.now()
      };

      // Inserisci nel DOM
      const inserted = this.insert(placeholder);
      if (inserted) {
        placeholder.inserted = true;
        stats.inserted++;

        // Track evento
        if (this.config.tracking.enabled && this.config.tracking.events.placeholderCreated) {
          this.trackEvent('placeholder_created', {
            id: placeholder.id,
            type: tag.type,
            dimensions: tag.dimensions
          });
        }
      }

      return placeholder;
    },

    /**
     * Applica stili al placeholder
     * @param {HTMLElement} element - Elemento placeholder
     * @param {Object} tag - Tag processato
     */
    applyStyles: function(element, tag) {
      // Base styles dalla config
      if (this.config.addBaseStyles) {
        element.setAttribute('style', this.config.placeholderBaseStyle);
      }

      // Stili specifici per formato
      const formatConfig = window.AdHubLegacyDefaults.getFormatConfig(tag.type);

      if (formatConfig && formatConfig.defaultStyle) {
        const currentStyle = element.getAttribute('style') || '';
        element.setAttribute('style', currentStyle + ' ' + formatConfig.defaultStyle);
      }

      // Imposta min-height
      if (formatConfig && formatConfig.minHeight) {
        const currentStyle = element.getAttribute('style') || '';
        element.setAttribute('style', currentStyle + ' min-height: ' + formatConfig.minHeight + ';');
      }

      // Per video: aggiungi attributi specifici
      if (tag.type === 'video' && formatConfig.videoAttributes) {
        Object.keys(formatConfig.videoAttributes).forEach(function(attr) {
          element.setAttribute(attr, formatConfig.videoAttributes[attr]);
        });
      }

      // Per responsive: aggiungi data-device
      if (this.config.responsive.autoHide) {
        const device = this.getDeviceType(tag.dimensionConfig);
        if (device) {
          element.setAttribute('data-device', device);
        }
      }
    },

    /**
     * Inserisce placeholder nel DOM
     * @param {Object} placeholder - Placeholder object
     * @returns {boolean} True se inserito con successo
     */
    insert: function(placeholder) {
      const tag = placeholder.tag;
      const element = placeholder.element;

      // Modalità inline: inserisci nella posizione dello script originale
      if (this.config.inlineInsertion && tag.scriptElement) {
        this.log('📍 Inserimento inline placeholder:', placeholder.id);

        try {
          // Inserisci dopo lo script
          if (tag.scriptElement.nextSibling) {
            tag.scriptElement.parentNode.insertBefore(
              element,
              tag.scriptElement.nextSibling
            );
          } else {
            tag.scriptElement.parentNode.appendChild(element);
          }

          // Opzionale: rimuovi o nascondi script originale
          if (!this.config.preserveOriginalScript) {
            tag.scriptElement.style.display = 'none';
          }

          this.log('✅ Placeholder inserito inline:', placeholder.id);
          return true;

        } catch (error) {
          this.log('❌ Errore inserimento inline:', error);
          // Fallback: append al body
        }
      }

      // Fallback: append al body
      this.log('📍 Inserimento placeholder nel body:', placeholder.id);

      try {
        document.body.appendChild(element);
        this.log('✅ Placeholder inserito nel body:', placeholder.id);
        return true;
      } catch (error) {
        this.log('❌ Errore inserimento nel body:', error);
        return false;
      }
    },

    /**
     * Determina tipo device dalla config dimensioni
     * @param {Object} dimensionConfig - Config dimensioni
     * @returns {string|null} Device type o null
     */
    getDeviceType: function(dimensionConfig) {
      if (!dimensionConfig || !dimensionConfig.category) {
        return null;
      }

      const categoryInfo = window.AdHubLegacyDimensions.getCategoryInfo(dimensionConfig.category);
      if (!categoryInfo || !categoryInfo.devices) {
        return null;
      }

      const devices = categoryInfo.devices;

      // Se supporta "all", ritorna null (nessun filtro)
      if (devices.includes('all')) {
        return null;
      }

      // Altrimenti ritorna lista devices separati da virgola
      return devices.join(',');
    },

    /**
     * Rimuovi placeholder dal DOM
     * @param {string} placeholderId - ID placeholder da rimuovere
     * @returns {boolean} True se rimosso
     */
    remove: function(placeholderId) {
      const element = document.getElementById(placeholderId);
      if (element && element.parentNode) {
        element.parentNode.removeChild(element);
        this.log('🗑️  Placeholder rimosso:', placeholderId);
        return true;
      }
      return false;
    },

    /**
     * Rimuovi tutti i placeholder legacy
     * @returns {number} Numero di placeholder rimossi
     */
    removeAll: function() {
      const placeholders = document.querySelectorAll('[data-adhub-legacy="true"]');
      let removed = 0;

      placeholders.forEach(function(element) {
        if (element.parentNode) {
          element.parentNode.removeChild(element);
          removed++;
        }
      });

      this.log('🗑️  Rimossi ' + removed + ' placeholder legacy');
      return removed;
    },

    /**
     * Verifica se placeholder esiste
     * @param {string} placeholderId - ID da verificare
     * @returns {boolean}
     */
    exists: function(placeholderId) {
      return !!document.getElementById(placeholderId);
    },

    /**
     * Ottieni elemento placeholder
     * @param {string} placeholderId - ID placeholder
     * @returns {HTMLElement|null}
     */
    getElement: function(placeholderId) {
      return document.getElementById(placeholderId);
    },

    /**
     * Ottieni tutti i placeholder legacy nel DOM
     * @returns {NodeList}
     */
    getAll: function() {
      return document.querySelectorAll('[data-adhub-legacy="true"]');
    },

    /**
     * Ottieni statistiche
     * @returns {Object}
     */
    getStats: function() {
      return {
        created: stats.created,
        inserted: stats.inserted,
        failed: stats.failed,
        successRate: stats.created > 0 ? ((stats.inserted / stats.created) * 100).toFixed(2) + '%' : '0%',
        placeholders: stats.placeholders
      };
    },

    /**
     * Reset statistiche
     */
    resetStats: function() {
      stats.created = 0;
      stats.inserted = 0;
      stats.failed = 0;
      stats.placeholders = [];
    },

    /**
     * Track evento
     */
    trackEvent: function(eventName, data) {
      if (!this.config.tracking.enabled) return;

      const event = {
        name: eventName,
        timestamp: Date.now(),
        data: data
      };

      this.log('📊 Track:', event);
    },

    /**
     * Helper per logging
     */
    log: function() {
      if (this.config && this.config.debug) {
        const args = Array.prototype.slice.call(arguments);
        console.log.apply(console, ['[AdHub Legacy Placeholder]'].concat(args));
      }
    },

    /**
     * Debug: mostra info placeholder
     */
    debug: function() {
      console.group('📦 AdHub Legacy Placeholder - Debug');
      console.log('Config:', this.config);
      console.log('Stats:', this.getStats());
      console.log('Placeholders in DOM:', this.getAll());
      console.groupEnd();
    }
  };

  // Esporta globalmente
  window.AdHubLegacyPlaceholder = AdHubLegacyPlaceholder;

})(window);


// ============================================================================
// src/core/loader.js
// ============================================================================

/**
 * AdHub Legacy SDK - Bundle Loader
 *
 * Carica dinamicamente il bundle AdHub dopo aver creato i placeholder
 * Gestisce retry e timeout
 *
 * @version 1.0.0
 */

(function(window) {
  'use strict';

  /**
   * Stato loader
   */
  const state = {
    loaded: false,
    loading: false,
    failed: false,
    attempts: 0,
    scriptElement: null,
    loadTime: null,
    error: null
  };

  /**
   * Loader principale
   */
  const AdHubLegacyLoader = {
    /**
     * Inizializza loader
     */
    init: function(config) {
      this.config = config || window.AdHubLegacyDefaults.getConfig();
      this.log('📡 Bundle Loader inizializzato');
    },

    /**
     * Carica bundle AdHub
     * @param {Function} callback - Callback chiamato dopo caricamento (success/fail)
     * @returns {Promise} Promise che si risolve quando bundle è caricato
     */
    load: function(callback) {
      // Se già caricato, chiama callback e ritorna
      if (state.loaded) {
        this.log('✅ Bundle già caricato');
        if (callback) callback(null, true);
        return Promise.resolve(true);
      }

      // Se già in loading, aspetta
      if (state.loading) {
        this.log('⏳ Bundle già in caricamento, attesa...');
        return this.waitForLoad(callback);
      }

      // Inizia caricamento
      this.log('📡 Caricamento bundle AdHub...');
      state.loading = true;
      state.attempts++;

      const startTime = Date.now();

      return new Promise(function(resolve, reject) {
        // Crea script element
        const script = document.createElement('script');
        script.type = 'text/javascript';
        script.async = true;

        // Costruisci URL con versioning se presente
        let bundleUrl = this.config.adhubBundleUrl;
        if (this.config.bundleVersion) {
          const separator = bundleUrl.indexOf('?') > -1 ? '&' : '?';
          bundleUrl += separator + 'v=' + this.config.bundleVersion;
        }

        script.src = bundleUrl;

        // Timeout handler
        const timeoutId = setTimeout(function() {
          this.log('⏱️  Timeout caricamento bundle');
          this.handleLoadError(new Error('Timeout'), callback, reject);
        }.bind(this), this.config.bundleLoadTimeout);

        // Success handler
        script.onload = function() {
          clearTimeout(timeoutId);

          state.loaded = true;
          state.loading = false;
          state.loadTime = Date.now() - startTime;

          this.log('✅ Bundle caricato con successo in ' + state.loadTime + 'ms');

          // Track evento
          if (this.config.tracking.enabled && this.config.tracking.events.bundleLoaded) {
            this.trackEvent('bundle_loaded', {
              url: this.config.adhubBundleUrl,
              loadTime: state.loadTime,
              attempts: state.attempts
            });
          }

          // Aspetta che AdHub sia pronto
          this.waitForAdHub(function() {
            if (callback) callback(null, true);
            resolve(true);
          });

        }.bind(this);

        // Error handler
        script.onerror = function() {
          clearTimeout(timeoutId);
          this.log('❌ Errore caricamento bundle');
          this.handleLoadError(new Error('Load failed'), callback, reject);
        }.bind(this);

        // Aggiungi script al DOM
        state.scriptElement = script;
        document.head.appendChild(script);

        this.log('📡 Script bundle aggiunto al DOM:', bundleUrl);

      }.bind(this));
    },

    /**
     * Gestisce errore caricamento
     * @param {Error} error - Errore
     * @param {Function} callback - Callback
     * @param {Function} reject - Promise reject
     */
    handleLoadError: function(error, callback, reject) {
      state.loading = false;
      state.error = error;

      // Se retry abilitato e non superato max tentativi
      if (this.config.retryBundleLoad && state.attempts < this.config.maxRetries) {
        this.log('🔄 Retry caricamento bundle (tentativo ' + (state.attempts + 1) + ')...');

        // Rimuovi script fallito
        if (state.scriptElement && state.scriptElement.parentNode) {
          state.scriptElement.parentNode.removeChild(state.scriptElement);
          state.scriptElement = null;
        }

        // Retry dopo delay
        setTimeout(function() {
          this.load(callback);
        }.bind(this), this.config.retryDelay);

        return;
      }

      // Nessun retry disponibile
      state.failed = true;

      this.log('❌ Caricamento bundle fallito definitivamente dopo ' + state.attempts + ' tentativi');

      // Track errore
      if (this.config.tracking.enabled && this.config.tracking.events.errors) {
        this.trackEvent('bundle_load_failed', {
          error: error.message,
          attempts: state.attempts,
          url: this.config.adhubBundleUrl
        });
      }

      if (callback) callback(error, false);
      if (reject) reject(error);
    },

    /**
     * Aspetta che bundle sia caricato (polling)
     * @param {Function} callback - Callback
     * @returns {Promise}
     */
    waitForLoad: function(callback) {
      return new Promise(function(resolve, reject) {
        const checkInterval = 100;
        const maxWait = this.config.bundleLoadTimeout;
        let waited = 0;

        const check = function() {
          if (state.loaded) {
            if (callback) callback(null, true);
            resolve(true);
            return;
          }

          if (state.failed) {
            const error = state.error || new Error('Load failed');
            if (callback) callback(error, false);
            reject(error);
            return;
          }

          waited += checkInterval;

          if (waited >= maxWait) {
            const error = new Error('Wait timeout');
            if (callback) callback(error, false);
            reject(error);
            return;
          }

          setTimeout(check, checkInterval);
        }.bind(this);

        check();
      }.bind(this));
    },

    /**
     * Aspetta che AdHub sia pronto e inizializzato
     * @param {Function} callback - Callback
     */
    waitForAdHub: function(callback) {
      this.log('⏳ Attesa inizializzazione AdHub...');

      const checkInterval = 100;
      const maxWait = 5000;
      let waited = 0;

      const check = function() {
        // Verifica che oggetti AdHub siano disponibili
        const adhubReady = (
          window.AdHubGAM &&
          window.AdHubAutoInit &&
          typeof pbjs !== 'undefined'
        );

        if (adhubReady) {
          this.log('✅ AdHub pronto');
          if (callback) callback();
          return;
        }

        waited += checkInterval;

        if (waited >= maxWait) {
          this.log('⚠️  Timeout attesa AdHub, procedo comunque');
          if (callback) callback();
          return;
        }

        setTimeout(check, checkInterval);
      }.bind(this);

      check();
    },

    /**
     * Verifica se bundle è caricato
     * @returns {boolean}
     */
    isLoaded: function() {
      return state.loaded;
    },

    /**
     * Verifica se bundle è in loading
     * @returns {boolean}
     */
    isLoading: function() {
      return state.loading;
    },

    /**
     * Verifica se caricamento è fallito
     * @returns {boolean}
     */
    hasFailed: function() {
      return state.failed;
    },

    /**
     * Ottieni stato loader
     * @returns {Object}
     */
    getState: function() {
      return {
        loaded: state.loaded,
        loading: state.loading,
        failed: state.failed,
        attempts: state.attempts,
        loadTime: state.loadTime,
        error: state.error ? state.error.message : null
      };
    },

    /**
     * Reset stato (per testing)
     */
    reset: function() {
      state.loaded = false;
      state.loading = false;
      state.failed = false;
      state.attempts = 0;
      state.scriptElement = null;
      state.loadTime = null;
      state.error = null;

      this.log('🔄 Stato loader resettato');
    },

    /**
     * Track evento
     */
    trackEvent: function(eventName, data) {
      if (!this.config.tracking.enabled) return;

      const event = {
        name: eventName,
        timestamp: Date.now(),
        data: data
      };

      this.log('📊 Track:', event);
    },

    /**
     * Helper per logging
     */
    log: function() {
      if (this.config && this.config.debug) {
        const args = Array.prototype.slice.call(arguments);
        console.log.apply(console, ['[AdHub Legacy Loader]'].concat(args));
      }
    },

    /**
     * Debug: mostra info loader
     */
    debug: function() {
      console.group('📡 AdHub Legacy Loader - Debug');
      console.log('Config:', this.config);
      console.log('State:', this.getState());
      console.log('Bundle URL:', this.config.adhubBundleUrl);
      console.groupEnd();
    }
  };

  // Esporta globalmente
  window.AdHubLegacyLoader = AdHubLegacyLoader;

})(window);


// ============================================================================
// src/sdk.js
// ============================================================================

/**
 * AdHub Legacy SDK - Main Entry Point
 *
 * SDK per retrocompatibilità con vecchi tag 4wnetwork
 * Rimpiazza: https://static.4wnetwork.com/js/sdk.min.js
 *
 * @version 1.0.0
 * @date 2025-11-17
 * @author AdHubMedia
 *
 * Funzionalità:
 * - Parse tag obj_4w legacy
 * - Mapping dimensioni → placeholder AdHub
 * - Generazione placeholder dinamici
 * - Caricamento automatico bundle AdHub
 * - Supporto Display, Native, Video ads
 * - Preserva targeting come key-values
 */

(function(window, document) {
  'use strict';

  // ============================================================================
  // PROTEZIONE DOPPIO CARICAMENTO
  // ============================================================================
  if (window.__AdHubLegacySDKLoaded) {
    console.warn('[AdHub Legacy SDK] SDK già caricato - Skip duplicato');
    return;
  }
  window.__AdHubLegacySDKLoaded = true;

  // ============================================================================
  // SDK PRINCIPALE
  // ============================================================================
  const AdHubLegacySDK = {
    // Versione
    version: '1.0.0',

    // Stato inizializzazione
    initialized: false,

    // Config
    config: null,

    // Componenti
    parser: null,
    placeholder: null,
    loader: null,

    // Risultati
    results: {
      tags: [],
      placeholders: [],
      errors: []
    },

    /**
     * Inizializzazione SDK
     * @param {Object} customConfig - Configurazione personalizzata
     */
    init: function(customConfig) {
      if (this.initialized) {
        console.warn('[AdHub Legacy SDK] SDK già inizializzato');
        return;
      }

      console.log('[AdHub Legacy SDK] 🚀 Inizializzazione v' + this.version + '...');

      // Carica config
      this.loadConfig(customConfig);

      // Debug mode check
      this.checkDebugMode();

      // Inizializza componenti
      this.initComponents();

      // Processa tag
      this.processTags();

      // Carica bundle AdHub
      this.loadAdHubBundle();

      this.initialized = true;
      console.log('[AdHub Legacy SDK] ✅ Inizializzazione completata');
    },

    /**
     * Carica configurazione
     * @param {Object} customConfig - Config custom
     */
    loadConfig: function(customConfig) {
      // Carica config da window se presente
      const windowConfig = window.AdHubLegacyDefaults.loadFromWindow();

      // Merge con config custom
      if (customConfig) {
        this.config = window.AdHubLegacyDefaults.mergeConfig(
          Object.assign({}, windowConfig, customConfig)
        );
      } else {
        this.config = windowConfig;
      }

      this.log('⚙️  Configurazione caricata:', this.config);
    },

    /**
     * Check debug mode da URL o config
     */
    checkDebugMode: function() {
      // Check URL parameter
      const urlParams = new URLSearchParams(window.location.search);
      if (urlParams.get('legacy_debug') === '1' || urlParams.get('debug') === 'legacy') {
        this.config.debug = true;
        console.log('[AdHub Legacy SDK] 🐛 Debug mode attivato da URL');
      }

      // Check window variable
      if (window.AdHubLegacyDebug === true) {
        this.config.debug = true;
        console.log('[AdHub Legacy SDK] 🐛 Debug mode attivato da window.AdHubLegacyDebug');
      }
    },

    /**
     * Inizializza componenti
     */
    initComponents: function() {
      this.log('🔧 Inizializzazione componenti...');

      // Parser
      if (window.AdHubLegacyParser) {
        this.parser = window.AdHubLegacyParser;
        this.parser.init(this.config);
        this.log('✅ Parser inizializzato');
      } else {
        console.error('[AdHub Legacy SDK] ❌ Parser non trovato!');
      }

      // Placeholder generator
      if (window.AdHubLegacyPlaceholder) {
        this.placeholder = window.AdHubLegacyPlaceholder;
        this.placeholder.init(this.config);
        this.log('✅ Placeholder generator inizializzato');
      } else {
        console.error('[AdHub Legacy SDK] ❌ Placeholder generator non trovato!');
      }

      // Loader
      if (window.AdHubLegacyLoader) {
        this.loader = window.AdHubLegacyLoader;
        this.loader.init(this.config);
        this.log('✅ Loader inizializzato');
      } else {
        console.error('[AdHub Legacy SDK] ❌ Loader non trovato!');
      }
    },

    /**
     * Processa tutti i tag obj_4w
     */
    processTags: function() {
      this.log('📋 Processamento tag...');

      if (!this.parser) {
        console.error('[AdHub Legacy SDK] ❌ Parser non disponibile');
        return;
      }

      // Parse tutti i tag
      const tags = this.parser.processAll();
      this.results.tags = tags;

      this.log('📊 Tag processati: ' + tags.length);

      if (tags.length === 0) {
        this.log('⚠️  Nessun tag da processare');
        return;
      }

      // Crea placeholder
      this.createPlaceholders(tags);
    },

    /**
     * Crea placeholder per i tag
     * @param {Array} tags - Tag processati
     */
    createPlaceholders: function(tags) {
      this.log('🏗️  Creazione placeholder...');

      if (!this.placeholder) {
        console.error('[AdHub Legacy SDK] ❌ Placeholder generator non disponibile');
        return;
      }

      // Crea tutti i placeholder
      const placeholders = this.placeholder.createAll(tags);
      this.results.placeholders = placeholders;

      this.log('📊 Placeholder creati: ' + placeholders.length);

      // Configura targeting per AdHub (se già caricato)
      this.configureTargeting(placeholders);
    },

    /**
     * Configura targeting per AdHub
     * @param {Array} placeholders - Placeholder creati
     */
    configureTargeting: function(placeholders) {
      this.log('🎯 Configurazione targeting...');

      // Se AdHub non è ancora caricato, salva per dopo
      if (!window.AdHubGAM) {
        this.log('⏳ AdHub non ancora caricato, targeting configurato dopo load');
        return;
      }

      // Configura targeting per ogni placeholder
      placeholders.forEach(function(placeholder) {
        const tag = placeholder.tag;

        if (!tag.targeting) return;

        this.log('🎯 Targeting per ' + placeholder.id + ':', tag.targeting);

        // TODO: Applicare targeting a GAM quando AdHub è pronto
        // Questo dipende dall'API di AdHubGAM

      }.bind(this));
    },

    /**
     * Carica bundle AdHub
     */
    loadAdHubBundle: function() {
      if (!this.config.autoLoadBundle) {
        this.log('⏭️  Auto-load bundle disabilitato');
        return;
      }

      if (!this.loader) {
        console.error('[AdHub Legacy SDK] ❌ Loader non disponibile');
        return;
      }

      this.log('📡 Caricamento bundle AdHub...');

      // Carica bundle
      this.loader.load(function(error, success) {
        if (error) {
          console.error('[AdHub Legacy SDK] ❌ Errore caricamento bundle:', error);
          this.results.errors.push({
            type: 'bundle_load',
            error: error.message
          });
          return;
        }

        this.log('✅ Bundle AdHub caricato con successo');

        // Configura targeting ora che AdHub è pronto
        this.configureTargeting(this.results.placeholders);

        // Dispatch evento custom
        this.dispatchReadyEvent();

      }.bind(this));
    },

    /**
     * Dispatch evento quando SDK è pronto
     */
    dispatchReadyEvent: function() {
      this.log('📢 Dispatch evento adhub:legacy:ready');

      const event = new CustomEvent('adhub:legacy:ready', {
        detail: {
          version: this.version,
          tags: this.results.tags.length,
          placeholders: this.results.placeholders.length,
          stats: this.getStats()
        }
      });

      window.dispatchEvent(event);
    },

    /**
     * Ottieni statistiche complete
     * @returns {Object} Statistiche
     */
    getStats: function() {
      return {
        sdk: {
          version: this.version,
          initialized: this.initialized
        },
        parser: this.parser ? this.parser.getStats() : null,
        placeholder: this.placeholder ? this.placeholder.getStats() : null,
        loader: this.loader ? this.loader.getState() : null,
        results: {
          tags: this.results.tags.length,
          placeholders: this.results.placeholders.length,
          errors: this.results.errors.length
        }
      };
    },

    /**
     * Ottieni config corrente
     * @returns {Object} Config
     */
    getConfig: function() {
      return this.config;
    },

    /**
     * Ottieni risultati processamento
     * @returns {Object} Risultati
     */
    getResults: function() {
      return this.results;
    },

    /**
     * Helper per logging
     */
    log: function() {
      if (this.config && this.config.debug) {
        const args = Array.prototype.slice.call(arguments);
        console.log.apply(console, ['[AdHub Legacy SDK]'].concat(args));
      }
    },

    /**
     * Debug completo
     */
    debug: function() {
      console.group('🔍 AdHub Legacy SDK - Complete Debug');
      console.log('Version:', this.version);
      console.log('Initialized:', this.initialized);
      console.log('Config:', this.config);
      console.log('Stats:', this.getStats());
      console.log('Results:', this.results);

      if (this.parser) {
        console.group('Parser');
        this.parser.debug();
        console.groupEnd();
      }

      if (this.placeholder) {
        console.group('Placeholder');
        this.placeholder.debug();
        console.groupEnd();
      }

      if (this.loader) {
        console.group('Loader');
        this.loader.debug();
        console.groupEnd();
      }

      if (window.AdHubLegacyDimensions) {
        console.group('Dimensions');
        window.AdHubLegacyDimensions.debug();
        console.groupEnd();
      }

      console.groupEnd();
    }
  };

  // ============================================================================
  // ESPORTA GLOBALMENTE
  // ============================================================================
  window.AdHubLegacySDK = AdHubLegacySDK;

  // ============================================================================
  // AUTO-INIT
  // ============================================================================
  function autoInit() {
    // Aspetta che DOM sia pronto
    if (document.readyState === 'loading') {
      document.addEventListener('DOMContentLoaded', function() {
        AdHubLegacySDK.init();
      });
    } else {
      // DOM già pronto, init immediato
      AdHubLegacySDK.init();
    }
  }

  // Avvia auto-init
  autoInit();

  console.log('[AdHub Legacy SDK] ✅ SDK caricato v' + AdHubLegacySDK.version);

})(window, document);

